using System;
using System.Data;
using System.Collections;
using Microsoft.Reporting.WinForms;
using System.Reflection;
using System.IO;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> 
	///WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified
	///</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Software Architects, Inc.</Developer>
	///</Developers>
	///	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>3/9/2004</CreationDate>
	///<Note>
	///The Food and Drug Administration classifies this software as a medical device.  
	///As such, it may not be changed in any way. Modifications to this software may 
	///result in an adulterated medical device under 21CFR820, the use of which is 
	///considered to be a violation of US Federal Statutes. Acquiring and implementing 
	///this software through the Freedom of information Act requires the implementor 
	///to assume total responsibility for the software, and become a registered 
	///manufacturer of a medical device, subject to FDA regulations.
	///</Note>
	///<summary>
	/// BloodAvailability business object
	///  
	/// 
	///</summary>

	#endregion

	public class BloodAvailability:VbecsReport
	{
		#region Variables

		private ArrayList _componentClasses;

		private ArrayList _aboRh;

		private ArrayList _donationTypes;

		private ArrayList _unitStatuses;

		private ArrayList _recordStatuses;

		private ArrayList _unitAntigens;

		private ArrayList _productAttributes;

		private ArrayList _divisions;

		private DateTime _expirationDateBefore;

		private DateTime _expirationDateAfter;

		private DateTime _finalDispositionStartDate;

		private DateTime _finalDispositionEndDate;

		private int _cmvTesting;

		private int _sickleCellTesting;

		private int _quarantined;

		private int _biohazardous;
		/// <summary>
		/// DataSet containing the items returned for a report.
		/// </summary>
		private DataSet _items = new DataSet("BloodAvailabilityItems");
		/// <summary>
		/// DataSet containing antigens.
		/// </summary>
		private DataSet _antigens = new DataSet("BloodAvailabilityAntigens");
		/// <summary>
		/// DataSet containing blood unit comments.
		/// </summary>
		private DataSet _comments = new DataSet("BloodAvailabilityComments");
		/// <summary>
		/// DataSet containing test comments.
		/// </summary>
		private DataSet _testcomments = new DataSet("BloodAvailabilityTestComments");
		/// <summary>
		/// DataSet containing test results.
		/// </summary>
		private DataSet _testresults = new DataSet("BloodAvailabilityTestComments");
		/// <summary>
		/// DataSet containing assigned patients
		/// </summary>
		private DataSet _assignedpatients = new DataSet("BloodAvailabilityAssignedPatients");
		/// <summary>
		/// DataSet containing target units
		/// </summary>
		private DataSet _targetUnits = new DataSet("BloodAvailabilityTargetUnits");
		/// <summary>
		/// Individual patient report patient
		/// </summary>
		private BOL.Patient _patient;

		private string _reportCriteria="";
		#endregion

		#region Constructors
		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3816"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>BloodAvailability object instantiated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3817"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// BloodAvailability
		/// </summary>
		public BloodAvailability()
		{
			this._aboRh=new ArrayList();
			this._componentClasses=new ArrayList();
			this._donationTypes=new ArrayList();
			this._unitAntigens=new ArrayList();
			this._unitStatuses=new ArrayList();
			this._productAttributes=new ArrayList();
			this._divisions=new ArrayList();
			this._recordStatuses=new ArrayList();
			this._expirationDateAfter=DateTime.Parse("1/1/1900");
			this._expirationDateBefore=DateTime.Parse("12/31/9999");
			this._finalDispositionStartDate=DateTime.Parse("1/1/1900");
			this._finalDispositionEndDate=DateTime.Parse("12/31/9999");
			this._cmvTesting=0;
			this._sickleCellTesting=0;
			this._biohazardous=0;
			this._quarantined=0;
			this._patient=new BOL.Patient();
		}

		#endregion
		
		#region Properties

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6187"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6188"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets or sets individual patient report indicator
		/// </summary>
		public bool IsIndividualPatientReport
		{
			get
			{
				if (this.ParameterRow.IsNull(ARTIFICIAL.IsIndividualPatientReport))
				{
					return false;
				}
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IsIndividualPatientReport]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IsIndividualPatientReport] = value;
				//
				RuleBroken("ReportType", (!this.IsIndividualPatientReport &&
					!this.IsCustomReport &&
					!this.IsDailyInventoryAllReport &&
					!this.IsDailyInventoryAvailableReport &&
					!this.IsAntigenTypedInventoryReport &&
					!this.IsNoDispositionReport &&
					!this.IsFinalDispositionReport));
				//
				RuleBroken("Patient", (this.IsIndividualPatientReport && (this.PatientGuid == Guid.Empty)));
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6189"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6190"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets or sets all units daily inventory report indicator
		/// </summary>
		public bool IsDailyInventoryAllReport
		{
			get
			{
				if (this.ParameterRow.IsNull(ARTIFICIAL.IsDailyInventoryAllReport))
				{
					return false;
				}
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IsDailyInventoryAllReport]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IsDailyInventoryAllReport] = value;
				//
				RuleBroken("ReportType", (!this.IsIndividualPatientReport &&
					!this.IsCustomReport &&
					!this.IsDailyInventoryAllReport &&
					!this.IsDailyInventoryAvailableReport &&
					!this.IsAntigenTypedInventoryReport &&
					!this.IsNoDispositionReport &&
					!this.IsFinalDispositionReport));
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6191"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6192"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets or sets available units daily inventory report indicator
		/// </summary>
		public bool IsDailyInventoryAvailableReport
		{
			get
			{
				if (this.ParameterRow.IsNull(ARTIFICIAL.IsDailyInventoryAvailableReport))
				{
					return false;
				}
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IsDailyInventoryAvailableReport]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IsDailyInventoryAvailableReport] = value;
				//
				RuleBroken("ReportType", (!this.IsIndividualPatientReport &&
					!this.IsCustomReport &&
					!this.IsDailyInventoryAllReport &&
					!this.IsDailyInventoryAvailableReport &&
					!this.IsAntigenTypedInventoryReport &&
					!this.IsNoDispositionReport &&
					!this.IsFinalDispositionReport));
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6193"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6194"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets or sets antigen typed unit inventory report indicator
		/// </summary>
		public bool IsAntigenTypedInventoryReport
		{
			get
			{
				if (this.ParameterRow.IsNull(ARTIFICIAL.IsAntigenTypedInventoryReport))
				{
					return false;
				}
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IsAntigenTypedInventoryReport]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IsAntigenTypedInventoryReport] = value;
				//
				RuleBroken("ReportType", (!this.IsIndividualPatientReport &&
					!this.IsCustomReport &&
					!this.IsDailyInventoryAllReport &&
					!this.IsDailyInventoryAvailableReport &&
					!this.IsAntigenTypedInventoryReport &&
					!this.IsNoDispositionReport &&
					!this.IsFinalDispositionReport));
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6195"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6196"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets or sets units with no disposition report indicator
		/// </summary>
		public bool IsNoDispositionReport
		{
			get
			{
				if (this.ParameterRow.IsNull(ARTIFICIAL.IsNoDispositionReport))
				{
					return false;
				}
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IsNoDispositionReport]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IsNoDispositionReport] = value;
				//
				RuleBroken("ReportType", (!this.IsIndividualPatientReport &&
					!this.IsCustomReport &&
					!this.IsDailyInventoryAllReport &&
					!this.IsDailyInventoryAvailableReport &&
					!this.IsAntigenTypedInventoryReport &&
					!this.IsNoDispositionReport &&
					!this.IsFinalDispositionReport));
			}
		}
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7362"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7363"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets units with final disposition report indicator
		/// </summary>
		public bool IsFinalDispositionReport
		{
			get
			{
				if (this.ParameterRow.IsNull(ARTIFICIAL.IsFinalDispositionReport))
				{
					return false;
				}
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IsFinalDispositionReport]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IsFinalDispositionReport] = value;
				//
				RuleBroken("ReportType", (!this.IsIndividualPatientReport &&
					!this.IsCustomReport &&
					!this.IsDailyInventoryAllReport &&
					!this.IsDailyInventoryAvailableReport &&
					!this.IsAntigenTypedInventoryReport &&
					!this.IsNoDispositionReport &&
					!this.IsFinalDispositionReport));
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6197"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6198"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets or sets custom report indicator
		/// </summary>
		public bool IsCustomReport
		{
			get
			{
				if (this.ParameterRow.IsNull(ARTIFICIAL.IsCustomReport))
				{
					return false;
				}
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IsCustomReport]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IsCustomReport] = value;
				//
				RuleBroken("ReportType", (!this.IsIndividualPatientReport &&
					!this.IsCustomReport &&
					!this.IsDailyInventoryAllReport &&
					!this.IsDailyInventoryAvailableReport &&
					!this.IsAntigenTypedInventoryReport &&
					!this.IsNoDispositionReport &&
					!this.IsFinalDispositionReport));
				//
				RuleBroken("Patient", (this.IsIndividualPatientReport && (this.PatientGuid == Guid.Empty)));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7364"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7365"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// FinalDispositionStartDate
		/// </summary>
		public DateTime FinalDispositionStartDate
		{
			get
			{
				return this._finalDispositionStartDate;
			}
			set
			{
				this._finalDispositionStartDate = value;
				//
				RuleBroken("FinalDispositionEndDate", (this.FinalDispositionEndDate < this.FinalDispositionStartDate));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7366"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7367"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// FinalDispositionEndDate
		/// </summary>
		public DateTime FinalDispositionEndDate
		{
			get
			{
				return this._finalDispositionEndDate;
			}
			set
			{
				this._finalDispositionEndDate = value;
				//
				RuleBroken(System.Reflection.MethodBase.GetCurrentMethod().Name, (this.FinalDispositionEndDate < this.FinalDispositionStartDate));
				//BR_77.11
				RuleBroken(ARTIFICIAL.PrintDate, (this.IsScheduled && (this.IsFinalDispositionReport || this.IsCustomReport) && ((this.PrintDate < BOL.VBECSDateTime.GetDivisionCurrentDateTime())||(this.PrintDate < this.FinalDispositionEndDate))));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7368"> 
		///		<ExpectedInput>Valid print date, IsScheduled = true</ExpectedInput>
		///		<ExpectedOutput>Updated blood availability object</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7369"> 
		///		<ExpectedInput>Valid print date, IsScheduled = false</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PrintDate
		/// </summary>
		public override DateTime PrintDate
		{
			get
			{
				if (!this.IsScheduled)
				{
					throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.Common.ReportNotScheduled().ResString);
				}
				return Convert.ToDateTime(base.PrintDate);
			}
			set
			{
				base.PrintDate=value; 
				//BR_77.11
				RuleBroken(System.Reflection.MethodBase.GetCurrentMethod().Name, (this.IsScheduled && (this.IsFinalDispositionReport || this.IsCustomReport) && ((base.PrintDate < BOL.VBECSDateTime.GetDivisionCurrentDateTime())||(base.PrintDate < this.FinalDispositionEndDate))));
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6199"> 
		///		<ExpectedInput>Valid patient Guid</ExpectedInput>
		///		<ExpectedOutput>Individual patient object set</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6200"> 
		///		<ExpectedInput>Empty patient Guid</ExpectedInput>
		///		<ExpectedOutput>Broken rules count increased</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets or sets individual patient report patient unique identifier
		/// </summary>
		public Guid PatientGuid
		{
			get
			{
				if (this.ParameterRow.IsNull(TABLE.Patient.PatientGuid))
				{
					return Guid.Empty;
				}
				return (Guid)this.ParameterRow[TABLE.Patient.PatientGuid];
			}
			set
			{
				this.ParameterRow[TABLE.Patient.PatientGuid] = value;
				//
				if (value != Guid.Empty)
				{
					this._patient = new BOL.Patient(value);
				}
				else
				{
					this._patient = null;
				}
				//
				RuleBroken("Patient", (this.IsIndividualPatientReport && (this.PatientGuid == Guid.Empty)));
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6201"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6202"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7103"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7104"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets or sets dynamic SQL filter statement -- For Custom Report ONLY
		/// </summary>
		public string CustomReportFilterStatement
		{
			get
			{
				if (!this.IsCustomReport)
				{
					throw new BOL.BusinessObjectException();
				}
				return this.ParameterRow[ARTIFICIAL.CustomReportFilterStatement].ToString();
			}
			set
			{
				if (!this.IsCustomReport)
				{
					throw new BOL.BusinessObjectException();
				}
				this.ParameterRow[ARTIFICIAL.CustomReportFilterStatement] = value;
			}
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3913"> 
		///		<ExpectedInput>Valid blood availability object</ExpectedInput>
		///		<ExpectedOutput>string detailing report criteria</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3914"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ReportCriteria
		/// </summary>
		public string ReportCriteria
		{
			get
			{
				this.BuildCriteriaList();
				return this._reportCriteria;
			}
		}

		#endregion

		#region Methods

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/21/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7146"> 
		///		<ExpectedInput>Valid blood availability object with filters</ExpectedInput>
		///		<ExpectedOutput>Updated blood availability object with no filters</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7147"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Clear all filters.
		/// </summary>
		public void ClearFilter()
		{
			this._componentClasses.Clear();
			this._divisions.Clear();
			this._aboRh.Clear();
			this._donationTypes.Clear();
			this._unitAntigens.Clear();
			this._unitStatuses.Clear();
			this._productAttributes.Clear();
			this._recordStatuses.Clear();
			this._cmvTesting=0;
			this._sickleCellTesting=0;
			this._biohazardous=0;
			this._quarantined=0;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3818"> 
		///		<ExpectedInput>Valid blood availability object</ExpectedInput>
		///		<ExpectedOutput>Count of units in daily inventory report</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3819"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="3820"> 
		///		<ExpectedInput>Valid blood availability object with filter statement defined</ExpectedInput>
		///		<ExpectedOutput>Count of units in custom report</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3821"> 
		///		<ExpectedInput>Blood availability object with NO filter statement defined</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception thrown</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="3825"> 
		///		<ExpectedInput>Valid blood availability object</ExpectedInput>
		///		<ExpectedOutput>Count of antigen typed units in report</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3826"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="3831"> 
		///		<ExpectedInput>Valid blood availability object</ExpectedInput>
		///		<ExpectedOutput>Count of units included in units with no disposition report</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3832"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8044"> 
		///		<ExpectedInput>Valid blood availability object</ExpectedInput>
		///		<ExpectedOutput>Count of units included in units with final disposition report</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="8045"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="3835"> 
		///		<ExpectedInput>Valid blood availability object with patient defined</ExpectedInput>
		///		<ExpectedOutput>Count of units included in the individual patient report</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3836"> 
		///		<ExpectedInput>Blood availability object with NO patient defined</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetBloodAvailabilityCount
		/// </summary>
		/// <returns></returns>
		public int GetBloodAvailabilityCount()
		{
			if (this.IsIndividualPatientReport)
			{
				if (this.PatientGuid == Guid.Empty)
				{
					throw new BOL.BusinessObjectException("Patient not defined");
				}
				return DAL.BloodAvailability.GetBloodAvailabilityPatientCount(this.PatientGuid);
			}
			//
			if (this.IsCustomReport)
			{
				if (this.CustomReportFilterStatement.Length==0)
				{
					throw new BOL.BusinessObjectException("Custom report filter not set");
				}
			}
			//
			string filter;
			//
			if (!this.IsCustomReport)
			{
				filter = this.BuildFilterStatement();
			}
			else
			{
				filter = this.CustomReportFilterStatement;
			}
			//
			return (int)(DAL.BloodAvailability.GetBloodAvailabilityCount(filter));
		}

		
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3863"> 
		///		<ExpectedInput>Valid blood availability object</ExpectedInput>
		///		<ExpectedOutput>Data set containing daily inventory report data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3864"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="3823"> 
		///		<ExpectedInput>Valid blood availability object with filter statement defined</ExpectedInput>
		///		<ExpectedOutput>Data set containing custom report data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3824"> 
		///		<ExpectedInput>Blood availability object with NO filter statement defined</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="3827"> 
		///		<ExpectedInput>Valid blood availability object</ExpectedInput>
		///		<ExpectedOutput>Data set containing antigen typed units report data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3828"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="3829"> 
		///		<ExpectedInput>Valid blood availability object</ExpectedInput>
		///		<ExpectedOutput>Data set containing units with no disposition report data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3830"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8046"> 
		///		<ExpectedInput>Valid blood availability object</ExpectedInput>
		///		<ExpectedOutput>Data set containing units with final disposition report data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8047"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="3833"> 
		///		<ExpectedInput>Valid blood availability object with patient defined</ExpectedInput>
		///		<ExpectedOutput>Data set containing individual patient report data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3834"> 
		///		<ExpectedInput>Blood availability object with NO patient defined</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7105"> 
		///		<ExpectedInput>Valid custom blood availability object with filter resulting in no data</ExpectedInput>
		///		<ExpectedOutput>Empty report data table</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetBloodAvailabilityReport
		/// </summary>
		/// <returns></returns>
		public DataSet GetBloodAvailabilityReport()
		{
			if (this.IsIndividualPatientReport)
			{
				if (this.PatientGuid == Guid.Empty)
				{
					throw new BOL.BusinessObjectException("Patient not defined");
				}
			}
			//
			if (this.IsCustomReport)
			{
				if (this.CustomReportFilterStatement.Length==0)
				{
					throw new BOL.BusinessObjectException("Custom report filter not set");
				}
			}
			//
			string filter;
			//
			if (!this.IsCustomReport)
			{
				filter = this.BuildFilterStatement();
			}
			else
			{
				filter = this.CustomReportFilterStatement;
			}
			// DataSet to hold the table of data.
			DataSet ds= new DataSet("BloodAvailabilityReport");

			// This function builds the 5 tables that are used to build the final table
			// is returned from this function.
			this.GetItems(filter);

			// Add the table to the current data set by building and passing it.
			ds.Tables.Add(BuildDataTable());

			return ds;
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/15/2004</CreationDate>
		///
		/// <summary>
		/// 
		/// </summary>
		private void BuildCriteriaList()
		{
			this._reportCriteria = "";
			//
			if(this._aboRh.Count>0)
			{
				string aborhfilter = "";
				foreach(string aborh in this._aboRh)
				{
					if (aborhfilter.Length >0)
					{
						aborhfilter+=", ";
					}
					aborhfilter+=aborh;
				}
				this._reportCriteria+="ABO/Rh: "+aborhfilter+"\n";
			}
			if(this._componentClasses.Count>0)
			{
				string classfilter = "";
				foreach(string componentclass in this._componentClasses)
				{
					DataRow drClass = BOL.ReferenceData.GetComponentClass(Convert.ToInt32(componentclass));
					if (classfilter.Length >0)
					{
						classfilter+=", ";
					}
					classfilter+= drClass[TABLE.ComponentClass.ComponentClassShortName].ToString();
				}
				this._reportCriteria+="Component Class: "+classfilter+"\n";
			}
			string testingfiiter = "";
			if(this._unitAntigens.Count>0)
			{
				foreach(string antigen in this._unitAntigens)
				{
					DataRow drAntigen = BOL.ReferenceData.GetAntigenType(Convert.ToInt32(antigen));
					if (testingfiiter.Length >0)
					{
						testingfiiter+=", ";
					}
					testingfiiter+=drAntigen[TABLE.AntigenType.AntigenTypeName].ToString();
				}
			}
			if (this._cmvTesting == 1)
			{
				if (testingfiiter.Length >0)
				{
					testingfiiter+=", ";
				}
				testingfiiter+="CMV Neg";
			}
			if (this._sickleCellTesting == 1)
			{
				if (testingfiiter.Length >0)
				{
					testingfiiter+=", ";
				}
				testingfiiter+="Sickle Cell Neg";
			}
			if (testingfiiter.Length>0)
			{
				this._reportCriteria+="Testing: "+testingfiiter+"\n";
			}
			//
			string statusfilter = "";
			if(this._unitStatuses.Count>0)
			{
				foreach(string status in this._unitStatuses)
				{
					DataRow drStatus = BOL.ReferenceData.GetUnitStatus(status);
					if (statusfilter.Length>0)
					{
						statusfilter+=", ";
					}
					statusfilter+=drStatus[TABLE.UnitStatus.UnitStatusText].ToString();
				}
			}
			else
			{
				//Non-final statuses used by default
				statusfilter+=Common.Utility.GetUnitStatusTextFromEnum(Common.UnitStatusCode.Limited) + ", ";
				statusfilter+=Common.Utility.GetUnitStatusTextFromEnum(Common.UnitStatusCode.Available) + ", ";
				statusfilter+=Common.Utility.GetUnitStatusTextFromEnum(Common.UnitStatusCode.Assigned) + ", ";
				statusfilter+=Common.Utility.GetUnitStatusTextFromEnum(Common.UnitStatusCode.CrossmatchedAssigned) + ", ";
				statusfilter+=Common.Utility.GetUnitStatusTextFromEnum(Common.UnitStatusCode.Issued);
			}
			//
			if (this._biohazardous==1)
			{
				if (statusfilter.Length>0)
				{
					statusfilter+=", ";
				}
				statusfilter+="Biohazardous";
			}
			if (this._quarantined==1)
			{
				if (statusfilter.Length>0)
				{
					statusfilter+=", ";
				}
				statusfilter+="Quarantined";
			}
			if (statusfilter.Length>0)
			{
				this._reportCriteria+="Unit Status: "+statusfilter+"\n";
			}
			if(this._donationTypes.Count>0)
			{
				string donationfilter = "";
				foreach(string donation in this._donationTypes)
				{
					if (donationfilter.Length>0)
					{
						donationfilter+=", ";
					}
					//
					if (donation == "A")
					{
						donationfilter+="Autologous";
					}
					else if (donation == "D")
					{
						donationfilter+="Directed";
					}
					else
					{
						donationfilter+="Allogeneic";
					}
				}
				this._reportCriteria+="Donation Type: "+donationfilter+"\n";
			}
			if(this._divisions.Count>0)
			{
				string divisionfilter = "";
				foreach(string division in this._divisions)
				{
					BOL.Division dvsn = new BOL.Division(division);
					if (divisionfilter.Length>0)
					{
						divisionfilter+=", ";
					}
					divisionfilter+=dvsn.DivisionName;
				}
				this._reportCriteria+="Division: "+divisionfilter+"\n";
			}
			if(this._productAttributes.Count>0)
			{
				string attfilter = "";
				foreach(string att in this._productAttributes)
				{
					DataRow drAttribute = BOL.ReferenceData.GetProductModification(att);
					if (attfilter.Length>0)
					{
						attfilter+=", ";
					}
					attfilter+=drAttribute[TABLE.ProductModification.ProductModificationText].ToString();
				}
				this._reportCriteria+="Product Attributes: "+attfilter+"\n";
			}
			if ((this._patient != null) && (!this._patient.IsNew))
			{
				this._reportCriteria+="Patient: " + this._patient.DisplayName+"\n";
			}
			if(this._expirationDateAfter>DateTime.Parse("1/1/1900"))
			{
				this._reportCriteria+="Expires After: "+Common.VBECSDateTime.FormatDateTimeString(this._expirationDateAfter) + "\n";
			}
			if(this._expirationDateBefore<DateTime.Parse("12/31/9999"))
			{
				this._reportCriteria+="Expires Before: "+Common.VBECSDateTime.FormatDateTimeString(this._expirationDateBefore) + "\n";
			}
			if (this.IsFinalDispositionReport||(this.IsCustomReport && this._finalDispositionStartDate>DateTime.Parse("1/1/1900")))
			{
				this._reportCriteria+="Final Disposition Start Date: "+Common.VBECSDateTime.FormatDateTimeString(this._finalDispositionStartDate) + "\n";
			}
			if (this.IsFinalDispositionReport||(this.IsCustomReport && this._finalDispositionEndDate<DateTime.Parse("12/31/9999")))
			{
				this._reportCriteria+="Final Disposition End Date: "+Common.VBECSDateTime.FormatDateTimeString(this._finalDispositionEndDate) + "\n";
			}
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3837"> 
		///		<ExpectedInput>String containing valid component class id</ExpectedInput>
		///		<ExpectedOutput>Component classes array list property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3838"> 
		///		<ExpectedInput>String containing component class id already in filter list</ExpectedInput>
		///		<ExpectedOutput>Component class array list not updated</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Used to add a new component class filter to the list
		/// of existing component class filters.
		/// </summary>
		/// <param name="filter">String</param>
		/// <returns></returns>
		public void AddComponentClassFilter(string filter)
		{
			if (this._componentClasses.Contains(filter))
			{
				return;
			}
			this._componentClasses.Add(filter);
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3839"> 
		///		<ExpectedInput>String containing valid ABO/Rh</ExpectedInput>
		///		<ExpectedOutput>ABO/Rh array list property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3840"> 
		///		<ExpectedInput>String containing ABO/Rh already in filter list</ExpectedInput>
		///		<ExpectedOutput>ABO/Rh array list not updated</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Used to add a new ABORh filter to the list
		/// of existing ABORh filters.
		/// </summary>
		/// <param name="filter">String</param>
		/// <returns></returns>
		public void AddABORhFilter(string filter)
		{
			if (this._aboRh.Contains(filter))
			{
				return;
			}
			this._aboRh.Add(filter);
		}
		
		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3841"> 
		///		<ExpectedInput>String containing valid donation type restriction id</ExpectedInput>
		///		<ExpectedOutput>Donation type array list property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3842"> 
		///		<ExpectedInput>String containing donation type already in filter list</ExpectedInput>
		///		<ExpectedOutput>Donation type array list not updated</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Used to add a new donation type filter to the list
		/// of existing donation type filters.
		/// </summary>
		/// <param name="filter">String</param>
		/// <returns></returns>
		public void AddDonationTypeFilter(string filter)
		{
			if (this._donationTypes.Contains(filter))
			{
				return;
			}
			this._donationTypes.Add(filter);
		}
		
		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3843"> 
		///		<ExpectedInput>String containing valid unit status code</ExpectedInput>
		///		<ExpectedOutput>Unit statuses array list property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3844"> 
		///		<ExpectedInput>String containing unit status already in filter list</ExpectedInput>
		///		<ExpectedOutput>Unit status array list not updated</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Used to add a new Unit Status filter to the list
		/// of existing Unit Status filters.
		/// </summary>
		/// <param name="filter">String</param>
		/// <returns></returns>
		public void AddUnitStatusFilter(string filter)
		{
			if (this._unitStatuses.Contains(filter))
			{
				return;
			}
			this._unitStatuses.Add(filter);
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3845"> 
		///		<ExpectedInput>String containing valid antigen type id</ExpectedInput>
		///		<ExpectedOutput>Unit antigen array list property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3846"> 
		///		<ExpectedInput>String containing unit antigen already in filter list</ExpectedInput>
		///		<ExpectedOutput>Unit antigen array list not updated</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Used to add a new Unit Antigen filter to the list
		/// of existing Unit Antigen filters.
		/// </summary>
		/// <param name="filter">String</param>
		/// <returns></returns>
		/// 
		public void AddUnitAntigenFilter(string filter)
		{
			if (this._unitAntigens.Contains(filter))
			{
				return;
			}
			this._unitAntigens.Add(filter);
		}
		
		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3847"> 
		///		<ExpectedInput>String containing valid product modification code</ExpectedInput>
		///		<ExpectedOutput>Product attribute array list property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3848"> 
		///		<ExpectedInput>String containing product modification code already in filter list</ExpectedInput>
		///		<ExpectedOutput>Product attribute array list not updated</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Used to add a new product attribute filter to the list
		/// of existing product attribute filters.
		/// </summary>
		/// <param name="filter">String</param>
		/// <returns></returns>
		public void AddProductAttributeFilter(string filter)
		{
			if (this._productAttributes.Contains(filter))
			{
				return;
			}
			this._productAttributes.Add(filter);
		}
		
		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3849"> 
		///		<ExpectedInput>String containing valid division code</ExpectedInput>
		///		<ExpectedOutput>Divisions array list property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3850"> 
		///		<ExpectedInput>String containing division code already in filter list</ExpectedInput>
		///		<ExpectedOutput>Divisions array list not updated</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Used to add a new Division filter to the list
		/// of existing division filters.
		/// </summary>
		/// <param name="filter">String</param>
		/// <returns></returns>
		public void AddDivisionFilter(string filter)
		{
			if (this._divisions.Contains(filter))
			{
				return;
			}
			this._divisions.Add(filter);
		}
		
		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3851"> 
		///		<ExpectedInput>Valid date/time</ExpectedInput>
		///		<ExpectedOutput>Expiration date range start date property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3852"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Sets the ExpirationDateBefore variable
		/// </summary>
		/// <param name="expirationDate">datetime</param>
		/// <returns></returns>
		public void AddExpirationDateBeforeFilter(DateTime expirationDate)
		{
			this._expirationDateBefore=expirationDate.Date.AddHours(23).AddMinutes(59).AddSeconds(59);
		}
		
		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3853"> 
		///		<ExpectedInput>Valid date/time</ExpectedInput>
		///		<ExpectedOutput>Expiration date range end date property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3854"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Sets the ExpirationDateAfter variable
		/// </summary>
		/// <param name="expirationDate">date time</param>
		/// <returns></returns>
		public void AddExpirationDateAfterFilter(DateTime expirationDate)
		{
			this._expirationDateAfter=expirationDate.Date;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7981"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7982"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Sets the FinalDispositionStartDateFilter varianle
		/// </summary>
		/// <param name="startDate"></param>
		public void AddFinalDispositionStartDateFilter(DateTime startDate)
		{
			this._finalDispositionStartDate = startDate.Date;
		}
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7370"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7371"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Sets the FinalDispositionEndDateFilter variable
		/// </summary>
		/// <param name="endDate">date time</param>
		/// <returns></returns>
		public void AddFinalDispositionEndDateFilter(DateTime endDate)
		{
			this._finalDispositionEndDate = endDate.Date.AddHours(23).AddMinutes(59).AddSeconds(59);
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3855"> 
		///		<ExpectedInput>Valid CMV Negative indicator (true)</ExpectedInput>
		///		<ExpectedOutput>CMV Testing property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3856"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Sets the CMVTesting variable value
		/// based upon the bool value passed in.
		/// </summary>
		/// <param name="TestResult">true-1, false-0</param>
		/// <returns></returns>
		public void AddCMVTestingFilter(bool TestResult)
		{
			if(TestResult)
				this._cmvTesting=1;
			else
				this._cmvTesting=0;
		}
		
		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3857"> 
		///		<ExpectedInput>Valid Sickle Cell Negative indicator (false)</ExpectedInput>
		///		<ExpectedOutput>Sickle Cell Testing property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3858"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Sets the SickleCellTesting variable value
		/// based upon the bool value passed in.
		/// </summary>
		/// <param name="TestResult">true-1, false-0</param>
		/// <returns></returns>
		public void AddSickleCellTestingFilter(bool TestResult)
		{
			if(TestResult)
				this._sickleCellTesting=1;
			else
				this._sickleCellTesting=0;
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="258"> 
		///		<ExpectedInput>Valid Quarantine indicator (true)</ExpectedInput>
		///		<ExpectedOutput>Quarantined property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="259"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Sets the CMVTesting variable value
		/// based upon the bool value passed in.
		/// </summary>
		/// <param name="filter">true-1, false-0</param>
		/// <returns></returns>
		public void AddQuarantineFilter(bool filter)
		{
			if(filter)
				this._quarantined=1;
			else
				this._quarantined=0;
		}
		
		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="260"> 
		///		<ExpectedInput>Valid Biohazardous indicator (false)</ExpectedInput>
		///		<ExpectedOutput>Biohazardous property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="261"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Sets the SickleCellTesting variable value
		/// based upon the bool value passed in.
		/// </summary>
		/// <param name="filter">true-1, false-0</param>
		/// <returns></returns>
		public void AddBiohazardousFilter(bool filter)
		{
			if(filter)
				this._biohazardous=1;
			else
				this._biohazardous=0;
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3907"> 
		///		<ExpectedInput>Valid patient object</ExpectedInput>
		///		<ExpectedOutput>Patient property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3908"> 
		///		<ExpectedInput>Invalid patient object</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///<summary>Adds a patient filter to this object.</summary>
		///<param name="p">Patient to add</param>
		public void AddPatientFilter(Patient p)
		{
			if ((p == null)||(p.PatientGuid == Guid.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("patient").ResString);
			}
			this.PatientGuid=p.PatientGuid;
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<summary>The function calls the DAL BloodAvailability functions
		///to populate items, antigens, comments, testcomments, and testresults
		///datasets</summary>
		///<param name="whereclause">string</param>
		private void GetItems( string whereclause )
		{
			// Items are the main body of the report, product id, name, patient name, etc...
			this._items=DAL.BloodAvailability.GetBloodAvailabilityItems(whereclause);
			
			// Antigens are any antigens assiged to blood units that are returned from the
			// search.
			this._antigens=DAL.BloodAvailability.GetBloodAvailabilityAntigens(whereclause);
			
			// Any comments about the blood units being searched on.
			this._comments=DAL.BloodAvailability.GetBloodAvailabilityComments(whereclause);
			
			// Any comments about any testing involved with the blood units.
			this._testcomments=DAL.BloodAvailability.GetBloodAvailabilityTestComments(whereclause);

			// Test results from tests performed on the blood units.
			this._testresults=DAL.BloodAvailability.GetBloodAvailabilityTestResults(whereclause);

			//Patient(s) assigned to the blood units
			//we have to strip out the patient restriction from the clause
			//because units can be assigned to more than one patient
			if (whereclause.IndexOf("pat.PatientGuid")>-1)
			{
				whereclause = whereclause.Replace(" AND  pat.PatientGuid='" + this._patient.PatientGuid.ToString() + "'","");
			}
			this._assignedpatients = DAL.BloodAvailability.GetBloodAvailabilityAssignedPatients(whereclause);

			// Target units of modified units.
			this._targetUnits = DAL.BloodAvailability.GetBloodAvailabilityTargetUnits(whereclause);
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3859"> 
		///		<ExpectedInput>Valid custom blood availabilty object</ExpectedInput>
		///		<ExpectedOutput>Filter string generated from blood availability properties</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7101"> 
		///		<ExpectedInput>Valid multi-divisional antigen typed blood availabilty object</ExpectedInput>
		///		<ExpectedOutput>Filter string generated from blood availability properties</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7102"> 
		///		<ExpectedInput>Valid daily inventory available blood availabilty object</ExpectedInput>
		///		<ExpectedOutput>Filter string generated from blood availability properties</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3860"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Builds a string that will be passed into the custom report
		/// builder function.
		/// </summary>
		/// <returns>
		/// String of boolean comparisons that is used to filter
		/// data from the stored procedure.
		/// </returns>
		public string BuildFilterStatement()
		{
			int nCount = 0;
			if (!this.IsCustomReport)
			{
				this.ClearFilter();
				//
				if(this.IsMultiDivisional)
				{
					foreach( BOL.DivisionDefinition divDef in BOL.DivisionDefinition.GetActiveVbecsDivisionsDefList() )
					{	
						if( divDef.DivisionCode.Substring(0,3) == Common.LogonUser.LogonUserDivisionCode.Substring(0,3) )
						{
							this.AddDivisionFilter( divDef.DivisionCode );
						}
					}					
				}
				else
				{	
					this.AddDivisionFilter(Common.LogonUser.LogonUserDivisionCode);
				}
				//
				if (this.IsAntigenTypedInventoryReport)
				{
					if(this.IsMultiDivisional==false)
					{
						this.AddUnitStatusFilter("A");
						this.AddUnitStatusFilter("L");
						this.AddUnitStatusFilter("S");
						this.AddUnitStatusFilter("C");
						this.AddExpirationDateAfterFilter(VBECSDateTime.GetDivisionCurrentDateTime());
					}
						// BR_49.18
					else
					{
						this.AddUnitStatusFilter("A");
						this.AddUnitStatusFilter("L");
						this.AddExpirationDateAfterFilter(VBECSDateTime.GetDivisionCurrentDateTime());
					}
					//
					DataTable datatable=BOL.Antigen.GetAntigenLookupList();
					foreach(DataRow div in datatable.Rows)
					{
						this.AddUnitAntigenFilter(div[TABLE.AntigenType.AntigenTypeId].ToString());
					}
				}
				//
				if (this.IsDailyInventoryAllReport)
				{
					// BR_49.11 & BR_49.16
					this.AddUnitStatusFilter("A");
					this.AddUnitStatusFilter("L");
					this.AddUnitStatusFilter("C");
					this.AddUnitStatusFilter("S");
				}
				//
				if (this.IsDailyInventoryAvailableReport)
				{
					// BR_49.12 && BR_49.17
					this.AddUnitStatusFilter("L");
					this.AddUnitStatusFilter("A");
				}
				//
				if (this.IsNoDispositionReport)
				{
					// BR_49.14
					AddUnitStatusFilter("A");
					AddUnitStatusFilter("L");
					this.AddExpirationDateBeforeFilter(VBECSDateTime.GetDivisionCurrentDateTime());
				}
				//
				if (this.IsFinalDispositionReport)
				{
					//BR_49.22
					AddUnitStatusFilter("D");
					AddUnitStatusFilter("M");
					AddUnitStatusFilter("X");
				}
				//
				if (this.IsIndividualPatientReport)
				{
					// BR_49.10 && BR_49.15
					this.AddUnitStatusFilter("S");
					this.AddUnitStatusFilter("C");
					this.AddUnitStatusFilter("I");
				}
			}
			//
			//array of strings to be populated with search criteria.
			string [] pars=new string [8];

			// Iterating through the ComponentClasses arraylist and 
			// creating a search string that will be added to seach string returned.
			pars[0] = "";
			if (this._componentClasses.Count>0)
			{
				nCount = this._componentClasses.Count;	
				pars[0] = "b.ComponentClassId IN (";
				for(int i=0; i<nCount; i++)
				{
					pars[0]+="'" + this._componentClasses[i].ToString() + "',";
				}
				pars[0]=pars[0].Substring(0,pars[0].Length-1) + ")";
			}

			// Iterating through the ABORh arraylist and 
			// creating a search string.
			pars[1] = "";
			if (this._aboRh.Count>0)
			{
				nCount = this._aboRh.Count;
				for(int i=0; i<nCount; i++)
				{
					switch(this._aboRh[i].ToString().Trim())
					{
						case "A":
							pars[1]+="(b.BloodTypeCode='A' AND b.RhFactorCode='Z')OR ";
							break;
						case "A Pos":
							pars[1]+="(b.BloodTypeCode='A' AND b.RhFactorCode='P')OR ";
							break;
						case "A Neg":
							pars[1]+="(b.BloodTypeCode='A' AND b.RhFactorCode='N')OR ";
							break;
						case "B":
							pars[1]+="(b.BloodTypeCode='B' AND b.RhFactorCode='Z')OR ";
							break;
						case "B Pos":
							pars[1]+="(b.BloodTypeCode='B' AND b.RhFactorCode='P')OR ";
							break;
						case "B Neg":
							pars[1]+="(b.BloodTypeCode='B' AND b.RhFactorCode='N')OR ";
							break;
						case "AB":
							pars[1]+="(b.BloodTypeCode='AB' AND b.RhFactorCode='Z')OR ";
							break;
						case "AB Pos":
							pars[1]+="(b.BloodTypeCode='AB' AND b.RhFactorCode='P')OR ";
							break;
						case "AB Neg":
							pars[1]+="(b.BloodTypeCode='AB' AND b.RhFactorCode='N')OR ";
							break;
						case "O":
							pars[1]+="(b.BloodTypeCode='O' AND b.RhFactorCode='Z')OR ";
							break;
						case "O Pos":
							pars[1]+="(b.BloodTypeCode='O' AND b.RhFactorCode='P')OR ";
							break;
						case "O Neg":
							pars[1]+="(b.BloodTypeCode='O' AND b.RhFactorCode='N')OR ";
							break;
						case "Pooled Pos":
							pars[1]+="(b.BloodTypeCode='Mx' AND b.RhFactorCode='P')OR ";
							break;
						case "Pooled Neg":
							pars[1]+="(b.BloodTypeCode='Mx' AND b.RhFactorCode='N')OR ";
							break;
						case "Pooled Pooled":
							pars[1]+="(b.BloodTypeCode='Mx' AND b.RhFactorCode='Y')OR ";
							break;
						case "O Pooled":
							pars[1]+="(b.BloodTypeCode='O' AND b.RhFactorCode='Y')OR ";
							break;
						case "A Pooled":
							pars[1]+="(b.BloodTypeCode='A' AND b.RhFactorCode='Y')OR ";
							break;
						case "B Pooled":
							pars[1]+="(b.BloodTypeCode='B' AND b.RhFactorCode='Y')OR ";
							break;
						case "AB Pooled":
							pars[1]+="(b.BloodTypeCode='AB' AND b.RhFactorCode='Y')OR ";
							break;
					}
				}
				if(pars[1]!=null && pars[1].Length>3)
					pars[1]=pars[1].Substring(0,pars[1].Length-3);
			}
	
			// Adding the antigens search criteria.
			pars[2] = "";
			if (this._unitAntigens.Count>0)
			{
				nCount = this._unitAntigens.Count;
				pars[2] = "a.AntigenTypeId IN (";
				for(int i=0; i<nCount; i++)
				{
					pars[2]+="'" + this._unitAntigens[i].ToString() + "',";
				}
				pars[2]=pars[2].Substring(0,pars[2].Length-1) + ")OR ";
			}
			// Adding the cmv testing and sickle cell testing criteria
			// to the parameters.
			if(this._cmvTesting==1)
				pars[2]+="(t.BloodTestTypeId=103 AND t.TestResultId='N' AND t.RecordStatusCode = 'A')OR ";
			if(this._sickleCellTesting==1)
				pars[2]+="(t.BloodTestTypeId=104 AND t.TestResultId='N' AND t.RecordStatusCode = 'A')OR ";
			if(pars[2].Length>3)
				pars[2]=pars[2].Substring(0,pars[2].Length-3);

			// Adding the product attributes criteria.
			pars[3] = "";
			if (this._productAttributes.Count>0)
			{
				nCount = this._productAttributes.Count;
				pars[3] = "(";
				for(int i=0; i<nCount; i++)
				{
					switch (Common.Utility.GetProductModificationCodeFromString(this._productAttributes[i].ToString()))
					{
						case Common.ProductModification.Irradiated:
							pars[3]+="(b.BloodUnitGuid IN (SELECT BloodUnitGuid FROM dbo.fnGetBloodUnitGuidsByProductModificationCode('I'))) OR";
							break;
						case Common.ProductModification.LeukoReduced:
							pars[3]+="(b.BloodUnitGuid IN (SELECT BloodUnitGuid FROM dbo.fnGetBloodUnitGuidsByProductModificationCode('L'))) OR";
							break;
						case Common.ProductModification.VolumeReduced:
							pars[3]+="(b.BloodUnitGuid IN (SELECT BloodUnitGuid FROM dbo.fnGetBloodUnitGuidsByProductModificationCode('V'))) OR";
							break;
						case Common.ProductModification.Deglycerolized:
							pars[3]+="(b.BloodUnitGuid IN (SELECT BloodUnitGuid FROM dbo.fnGetBloodUnitGuidsByProductModificationCode('D'))) OR";
							break;
						case Common.ProductModification.Thawed:
							pars[3]+="(b.BloodUnitGuid IN (SELECT BloodUnitGuid FROM dbo.fnGetBloodUnitGuidsByProductModificationCode('T'))) OR";
							break;
						case Common.ProductModification.Frozen:
							pars[3]+="(b.BloodUnitGuid IN (SELECT BloodUnitGuid FROM dbo.fnGetBloodUnitGuidsByProductModificationCode('F'))) OR";
							break;
						case Common.ProductModification.Pooled:
							pars[3]+="(b.BloodUnitGuid IN (SELECT BloodUnitGuid FROM dbo.fnGetBloodUnitGuidsByProductModificationCode('P'))) OR";
							break;
						case Common.ProductModification.Rejuvenated:
							pars[3]+="(b.BloodUnitGuid IN (SELECT BloodUnitGuid FROM dbo.fnGetBloodUnitGuidsByProductModificationCode('R'))) OR";
							break;
						case Common.ProductModification.PoolThawCryo:
							pars[3]+="(b.BloodUnitGuid IN (SELECT BloodUnitGuid FROM dbo.fnGetBloodUnitGuidsByProductModificationCode('U'))) OR";
							break;
						case Common.ProductModification.Washed:
							pars[3]+="(b.BloodUnitGuid IN (SELECT BloodUnitGuid FROM dbo.fnGetBloodUnitGuidsByProductModificationCode('W'))) OR";
							break;
						case Common.ProductModification.SplitAdult:
							pars[3]+="(b.BloodUnitGuid IN (SELECT BloodUnitGuid FROM dbo.fnGetBloodUnitGuidsByProductModificationCode('S'))) OR";
							break;
						default:
							break;
					}
				}
				pars[3]=pars[3].Substring(0,pars[3].Length-3) + ")";
			}

			// Adding the divisions search criteria.
			pars[4] = "";
			if (this._divisions.Count>0)
			{
				nCount = this._divisions.Count;
				pars[4] = "b.DivisionCode IN (";
				for(int i=0; i<nCount; i++)
				{
					pars[4]+="'" + this._divisions[i].ToString().Trim() + "',";
				}
				pars[4]=pars[4].Substring(0,pars[4].Length-1) + ")";
			}

			// Adding the donation type search criteria.
			pars[5] = "";
			if (this._donationTypes.Count>0)
			{
				nCount = this._donationTypes.Count;
				pars[5] = "dt.RestrictionTypeCode IN (";
				for(int i=0; i<nCount; i++)
				{
					pars[5]+="'" + this._donationTypes[i].ToString() + "',";
				}
				pars[5]=pars[5].Substring(0,pars[5].Length-1) + ")";
			}

			// Adding the unit status search criteria.
			pars[6] = "";
			if (this._unitStatuses.Count==0)
			{
				//Custom report where no unit status was specified.. therefore return all non-final statuses
				pars[6]+="b.UnitStatusCode<>'"+Common.Utility.GetUnitStatusCodeCharFromEnum(Common.UnitStatusCode.Discarded).ToString() + "' AND ";
				pars[6]+="b.UnitStatusCode<>'"+Common.Utility.GetUnitStatusCodeCharFromEnum(Common.UnitStatusCode.Modified).ToString() + "' AND ";
				pars[6]+="b.UnitStatusCode<>'"+Common.Utility.GetUnitStatusCodeCharFromEnum(Common.UnitStatusCode.Transferred).ToString() + "' AND ";
				pars[6]+="b.UnitStatusCode<>'"+Common.Utility.GetUnitStatusCodeCharFromEnum(Common.UnitStatusCode.Transfused).ToString() + "'OR ";
			}
			else
			{
				nCount = this._unitStatuses.Count;
				pars[6] = "b.UnitStatusCode IN (";
				for(int i=0; i<nCount; i++)
				{
					pars[6]+="'" + this._unitStatuses[i].ToString() + "',";
				}
				pars[6]=pars[6].Substring(0,pars[6].Length-1) + ")OR ";
			}
			// If quarantined, add to parameters.
			if(this._quarantined==1)
				pars[6]+=" b.QuarantineIndicator=1 OR ";
			// If biohazardous, add to parameters.
			if(this._biohazardous==1)
				pars[6]+=" b.BiohazardousWarningIndicator=1 OR ";
			if(pars[6]!=null && pars[6].Length>3)
				pars[6]=pars[6].Substring(0,pars[6].Length-3);

			pars[7] = "";
			// If a patient filter has been added, add to the criteria.
			if ((this._patient != null) && (!this._patient.IsNew))
				pars[7] = " pat.PatientGuid='" + this._patient.PatientGuid.ToString() + "' AND ";
		
			// ************************************************************************************************************************************************
			// CR 2880 - Need to do time zone conversions for the dates
			string timeZoneConvert = "dbo.fnTimeZoneConversionDateTime('" + Common.LogonUser.LogonUserDivisionCode + "', 0, ";

			// Adding dates to the search. These will always have a value and are always passed to the stored procedure.
			pars[7] += timeZoneConvert + "b.UnitExpirationDate) < '" + this._expirationDateBefore.ToString() + "' AND ";
			pars[7] += timeZoneConvert + "b.UnitExpirationDate) > '" + this._expirationDateAfter.ToString() + "'";

			if ((this.IsFinalDispositionReport)||(this.IsCustomReport))
			{
				pars[7] += " AND(b.UnitStatusCode NOT IN ('D','M','X') OR ";

				// Adding dates to the search. These will always have a value and are always passed to the stored procedure.
				pars[7] += "(b.UnitStatusCode='D' AND " + timeZoneConvert + "s.UnitStatusDate) > '" + this._finalDispositionStartDate.ToString() 
							+ "' AND " + timeZoneConvert + "s.UnitStatusDate) < '" + this._finalDispositionEndDate.ToString() + "') OR ";
				pars[7] += "(b.UnitStatusCode='X' AND " + timeZoneConvert + "os.ShipmentDate) > '" + this._finalDispositionStartDate.ToString() 
							+ "' AND " + timeZoneConvert + "os.ShipmentDate) < '" + this._finalDispositionEndDate.ToString() + "') OR ";
				pars[7] += "(b.UnitStatusCode='M' AND " + timeZoneConvert + "d.ModificationDate) > '" + this._finalDispositionStartDate.ToString() 
							+ "' AND " + timeZoneConvert + "d.ModificationDate) < '" + this._finalDispositionEndDate.ToString() + "'))";
			}
			// ************************************************************************************************************************************************

			// Parameters string which is going to be returned. It will
			// contain all of the search criteria necessary to for
			// this reporting function to work.
			System.Text.StringBuilder parameters = new System.Text.StringBuilder(4000, 4000);
			// Return string filled with search parameters.
			// If any of the strings in pars have not been populated
			// do not include them in the parameters. If they have 
			// been populated, add to parameters.
			if(pars[0]!="" && pars[0]!=null)
				parameters.Append("("+pars[0]+") AND ");
			if(pars[1]!="" && pars[1]!=null)
				parameters.Append("("+pars[1]+") AND ");
			if(pars[2]!="" && pars[2]!=null)
				parameters.Append("("+pars[2]+") AND ");
			if(pars[3]!="" && pars[3]!=null)
				parameters.Append("("+pars[3]+") AND ");
			if(pars[4]!="" && pars[4]!=null)
				parameters.Append("("+pars[4]+") AND ");
			if(pars[5]!="" && pars[5]!=null)
				parameters.Append("("+pars[5]+") AND ");
			if(pars[6]!="" && pars[6]!=null)
				parameters.Append("("+pars[6]+") AND ");
			if(pars[7]!="" && pars[7]!=null)
				parameters.Append("("+pars[7]+")");
			return parameters.ToString();
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		/// <summary>
		/// Implements BR_49.09
		/// </summary>
		/// <returns></returns>
		private DataTable BuildDataTable()
		{
			// Table to hold data for the report 
			// to be returned.
			DataTable dt = new DataTable("");

			// Building the table layout.
			dt.Columns.Add(TABLE.VamcDivision.DivisionCode);
			dt.Columns.Add(TABLE.VamcDivision.DivisionName);
			dt.Columns.Add(ARTIFICIAL.ThisDivision, typeof(bool));
			dt.Columns.Add(ARTIFICIAL.UnitIdentification);
			dt.Columns.Add(TABLE.ComponentClass.ComponentClassId);
			dt.Columns.Add(ARTIFICIAL.ComponentClassSortColumn, typeof(int));
			dt.Columns.Add(TABLE.BloodProduct.ProductName);
			dt.Columns.Add(ARTIFICIAL.ABORh);
			dt.Columns.Add(ARTIFICIAL.ABOSortColumn, typeof(int));
			dt.Columns.Add(ARTIFICIAL.RhSortColumn, typeof(int));
			dt.Columns.Add(ARTIFICIAL.UnitExpiration, typeof(DateTime));
			dt.Columns.Add(ARTIFICIAL.DonationTypes);
			dt.Columns.Add(ARTIFICIAL.UnitStatus);
			dt.Columns.Add(ARTIFICIAL.UnitStatusSortColumn, typeof(int));
			dt.Columns.Add(ARTIFICIAL.AssignedPatients);
			dt.Columns.Add(ARTIFICIAL.AntigenTyping);
			dt.Columns.Add(ARTIFICIAL.AdditionalInformation);
			dt.Columns.Add(ARTIFICIAL.PatientReportName);
			dt.Columns.Add(ARTIFICIAL.PatientReportId);
			dt.Columns.Add(ARTIFICIAL.NoDataIndicator, typeof(bool));
			
			// If the number of rows returned is more than 1
			if (this._items.Tables[0].Rows.Count>0)
			{
				// iterate through the items table populating the
				// data table that is going to be returned.
				for(int i=0; i<this._items.Tables[0].Rows.Count; i++)
				{	
					// New row to populate with data and add to the data
					// table that is going to be returned.
					DataRow dr=dt.NewRow();
					dr[TABLE.VamcDivision.DivisionCode]=this._items.Tables[0].Rows[i][TABLE.VamcDivision.DivisionCode].ToString();
					dr[TABLE.VamcDivision.DivisionName]=this._items.Tables[0].Rows[i][TABLE.VamcDivision.DivisionName].ToString();
					dr[ARTIFICIAL.ThisDivision]=(this._items.Tables[0].Rows[i][TABLE.VamcDivision.DivisionCode].ToString().Trim()==Common.LogonUser.LogonUserDivisionCode.Trim());
					dr[ARTIFICIAL.UnitIdentification]=this._items.Tables[0].Rows[i][TABLE.BloodUnit.EyeReadableUnitId].ToString();
					dr[TABLE.ComponentClass.ComponentClassId]=this._items.Tables[0].Rows[i][TABLE.ComponentClass.ComponentClassId].ToString();
					dr[ARTIFICIAL.ComponentClassSortColumn]=Convert.ToInt16(this._items.Tables[0].Rows[i][ARTIFICIAL.ComponentClassSortColumn]);
					dr[TABLE.BloodProduct.ProductName]=this._items.Tables[0].Rows[i][TABLE.BloodProduct.ProductShortName].ToString();
					if (!this._items.Tables[0].Rows[i].IsNull(ARTIFICIAL.PooledUnitsCount))
					{
						if (Convert.ToInt32(this._items.Tables[0].Rows[i][ARTIFICIAL.PooledUnitsCount])>0)
						{
							dr[TABLE.BloodProduct.ProductName] += " x" + this._items.Tables[0].Rows[i][ARTIFICIAL.PooledUnitsCount].ToString();
						}
					}
					BOL.AboRh aborh = new BOL.AboRh(Common.Utility.GetAboFromString(this._items.Tables[0].Rows[i][TABLE.BloodType.BloodTypeCode].ToString()),Common.Utility.GetRhFromString(this._items.Tables[0].Rows[i][TABLE.RhFactor.RhFactorCode].ToString()));
					dr[ARTIFICIAL.ABORh] = aborh.AboRHText;
					dr[ARTIFICIAL.ABOSortColumn] = Convert.ToInt16(this._items.Tables[0].Rows[i][ARTIFICIAL.ABOSortColumn]);
					dr[ARTIFICIAL.RhSortColumn] = Convert.ToInt16(this._items.Tables[0].Rows[i][ARTIFICIAL.RhSortColumn]);
					dr[ARTIFICIAL.UnitExpiration]=(DateTime)this._items.Tables[0].Rows[i][TABLE.BloodUnitMedia.UnitExpirationDate];
					dr[ARTIFICIAL.DonationTypes]=this._items.Tables[0].Rows[i][TABLE.DonationType.DonationTypeCode].ToString();
					dr[ARTIFICIAL.UnitStatus]=this._items.Tables[0].Rows[i][TABLE.UnitStatus.UnitStatusText].ToString();
					dr[ARTIFICIAL.UnitStatusSortColumn] = Convert.ToInt16(this._items.Tables[0].Rows[i][ARTIFICIAL.UnitStatusSortColumn]);
					if ((bool)this._items.Tables[0].Rows[i][TABLE.BloodUnitStatus.BiohazardousWarningIndicator])
					{
						dr[ARTIFICIAL.UnitStatus]=dr[ARTIFICIAL.UnitStatus]+" (Biohazardous)";
					}
					if ((bool)this._items.Tables[0].Rows[i][TABLE.BloodUnitStatus.QuarantineIndicator])
					{
						dr[ARTIFICIAL.UnitStatus]=dr[ARTIFICIAL.UnitStatus]+" (Quarantined)";
					}

					//Adding the assigned patients to the table
					string assignedPatients = "";
					for (int j=0; j<this._assignedpatients.Tables[0].Rows.Count; j++)
					{
						if(this._items.Tables[0].Rows[i][TABLE.BloodUnit.BloodUnitGuid].Equals(this._assignedpatients.Tables[0].Rows[j][TABLE.BloodUnit.BloodUnitGuid]))
						{
							if (assignedPatients.Length>0)
							{
								assignedPatients += "\n";
							}
							string firstname = this._assignedpatients.Tables[0].Rows[j][TABLE.Patient.PatientFirstName].ToString();
							string middlename = this._assignedpatients.Tables[0].Rows[j][TABLE.Patient.PatientMiddleName].ToString();
							string lastname = this._assignedpatients.Tables[0].Rows[j][TABLE.Patient.PatientLastName].ToString();
							string patientid = this._assignedpatients.Tables[0].Rows[j][TABLE.Patient.PatientSsn].ToString();
							string displayname = BOL.Patient.BuildDisplayName(firstname, middlename, lastname);
							assignedPatients += displayname+"\n"+patientid;
						}
					}
					dr[ARTIFICIAL.AssignedPatients] = assignedPatients;

					// Adding the antigens to the table
					string antigen=string.Empty;
					for(int j=0; j<this._antigens.Tables[0].Rows.Count; j++)
					{
						if(this._items.Tables[0].Rows[i][TABLE.BloodUnit.BloodUnitGuid].Equals(this._antigens.Tables[0].Rows[j][TABLE.BloodUnit.BloodUnitGuid]))
						{
							if (antigen.Length>0)
							{
								antigen += "\n";
							}
							antigen+=this._antigens.Tables[0].Rows[j][TABLE.AntigenType.AntigenTypeName].ToString();
						}
					}
					dr[ARTIFICIAL.AntigenTyping]=antigen;

					// Gathering data for the additional info column of the report
					// Data comes from the comments, test comments, and test results
					// data sets returned.
					string additionalInfo=string.Empty;
					for(int j=0; j<this._comments.Tables[0].Rows.Count; j++)
					{
						if(this._items.Tables[0].Rows[i][TABLE.BloodUnit.BloodUnitGuid].Equals(this._comments.Tables[0].Rows[j][TABLE.BloodUnit.BloodUnitGuid]))
						{
							additionalInfo=additionalInfo+"Comments: " + this._comments.Tables[0].Rows[j][TABLE.BloodUnitMedia.Comments].ToString() + " \n";
						}
					}
					if (this._items.Tables[0].Rows[i][TABLE.TestingStatus.TestingStatusText].ToString().Length>0)
					{
						additionalInfo+=this._items.Tables[0].Rows[i][TABLE.TestingStatus.TestingStatusText].ToString() + "\n";
					}
					for(int j=0; j<this._testresults.Tables[0].Rows.Count; j++)
					{
						if(this._items.Tables[0].Rows[i][TABLE.BloodUnit.BloodUnitGuid].Equals(this._testresults.Tables[0].Rows[j][TABLE.BloodUnit.BloodUnitGuid]))
						{
							if(this._testresults.Tables[0].Rows[j][TABLE.TestResult.TestResultText].ToString().Length>0)
							{
								additionalInfo=additionalInfo
									+this._testresults.Tables[0].Rows[j][TABLE.BloodTestType.BloodTestName].ToString() + " "
									+this._testresults.Tables[0].Rows[j][TABLE.TestResult.TestResultText].ToString() + " \n";
							}
						}
					}
					if (!this._items.Tables[0].Rows[i].IsNull(TABLE.BloodUnitMedia.DirectedPatientGuid))
					{
						additionalInfo+="Restricted To: "+BOL.Patient.BuildDisplayName(this._items.Tables[0].Rows[i][ARTIFICIAL.DirectedPatientFirstName].ToString(),
							this._items.Tables[0].Rows[i][ARTIFICIAL.DirectedPatientMiddleName].ToString(),
							this._items.Tables[0].Rows[i][ARTIFICIAL.DirectedPatientLastName].ToString());
					}
					if (this._items.Tables[0].Rows[i][TABLE.IssuedUnit.IssueToLocationName].ToString().Length>0)
					{
						additionalInfo+="Issued To: " + this._items.Tables[0].Rows[i][TABLE.IssuedUnit.IssueToLocationName].ToString();
					}
					for(int j=0; j<this._targetUnits.Tables[0].Rows.Count; j++)
					{
						if(this._items.Tables[0].Rows[i][TABLE.BloodUnit.BloodUnitGuid].Equals(this._targetUnits.Tables[0].Rows[j][TABLE.BloodUnit.BloodUnitGuid]))
						{
							additionalInfo=additionalInfo+"Target unit: " + this._targetUnits.Tables[0].Rows[j][TABLE.BloodUnit.EyeReadableUnitId].ToString() + ", " + this._targetUnits.Tables[0].Rows[j][TABLE.BloodProduct.ProductShortName].ToString() + " \n";
						}
					}
					// Adding additional information to the data table.
					dr[ARTIFICIAL.AdditionalInformation]=additionalInfo;
					
					//Add Patient Name & ID if this is an individual patient report
					if (this._patient != null)
					{
						dr[ARTIFICIAL.PatientReportName] = this._patient.DisplayName;
						dr[ARTIFICIAL.PatientReportId] = this._patient.DisplayPatientID;
					}
					//
					dr[ARTIFICIAL.NoDataIndicator] = false;
					//
					// Adding a new row to the table.
					dt.Rows.Add(dr);
				}
				if (this.IsMultiDivisional)
				{
					//Create a blank row for any division that does not have a record
					//
					VbecsUser user = new BOL.VbecsUser(this.PrintedBy);
					foreach(BOL.Division division in user.DivisionsAndRoles.EffectiveDivisions)
					{
						DataRow[] drDivision = dt.Select(TABLE.VamcDivision.DivisionCode + " = '" + division.DivisionCode + "'");
						if (drDivision.Length == 0)
						{
							//
							foreach(DataColumn col in dt.Columns)
							{
								col.AllowDBNull = true;
							}
							//this division is not represented in the report
							DataRow drBlank = dt.NewRow();
							drBlank[TABLE.VamcDivision.DivisionCode] = division.DivisionCode;
							drBlank[TABLE.VamcDivision.DivisionName] = division.DivisionName;
							drBlank[ARTIFICIAL.NoDataIndicator] = true;
							dt.Rows.Add(drBlank);
						}
					}
				}
				//
				//BR_49.09
				DataRow[] drSort = dt.Select("","ThisDivision DESC, DivisionName, ComponentClassSortColumn, ABOSortColumn, RhSortColumn, UnitStatusSortColumn, ProductName, UnitExpiration, UnitIdentification");
				DataTable dtReturn = dt.Clone();
				foreach(DataRow dr in drSort)
				{
					dtReturn.Rows.Add(dr.ItemArray);
				}
				//
				return dtReturn;
			}
			else
				// return the data table with no data.
				return dt;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6060"> 
		///		<ExpectedInput>IsDailyInventoryAllReport = true</ExpectedInput>
		///		<ExpectedOutput>Daily inventory (all) report</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8159"> 
		///		<ExpectedInput>IsAntigenTypedInventoryReport = true</ExpectedInput>
		///		<ExpectedOutput>Antigen typed report</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8160"> 
		///		<ExpectedInput>IsDailyInventoryAvailableReport = true</ExpectedInput>
		///		<ExpectedOutput>Daily inventory (available) report</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8161"> 
		///		<ExpectedInput>IsFinalDispositionReport = true</ExpectedInput>
		///		<ExpectedOutput>Units in final disposition report</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8162"> 
		///		<ExpectedInput>IsIndividualPatientReport = true</ExpectedInput>
		///		<ExpectedOutput>Individual patient report</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8163"> 
		///		<ExpectedInput>IsNoDispositionReport = true</ExpectedInput>
		///		<ExpectedOutput>Units with no disposition report</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8164"> 
		///		<ExpectedInput>IsCustomReport = true</ExpectedInput>
		///		<ExpectedOutput>Custom blood availability report</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6061"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GenerateReport
		/// </summary>
		/// <returns></returns>
		public string GenerateReportTitle()
		{
			//	DataSet dsReport = this.GetBloodAvailabilityReport();
			//
#if DEBUG
			//This is justified dead code.
			//Uncomment if you need to regenerate XML schema for this report
			//dsReport.WriteXml("c:\\BloodAvailabilityReport.xml", XmlWriteMode.WriteSchema);
#endif
			//
			string reportTitle = "Blood Availability Report";
			//
			if (this.IsAntigenTypedInventoryReport)
			{
				reportTitle = "Antigen Typed Inventory (Available Units)";
			}
			else if (this.IsDailyInventoryAvailableReport)
			{
				reportTitle = "Daily Inventory (Available Units)";
			}
			else if (this.IsDailyInventoryAllReport)
			{
				reportTitle = "Daily Inventory (All Units)";
			}
			else if (this.IsIndividualPatientReport)
			{
				reportTitle = "Patient Blood Availability Report";
			}
			else if (this.IsNoDispositionReport)
			{
				reportTitle = "Units With No Disposition";
			}
			else if (this.IsFinalDispositionReport)
			{
				reportTitle = "Units With Final Disposition";
			}	
			
			/*gov.va.med.vbecs.reports.BloodAvailability.BloodAvailabilityReport report = new gov.va.med.vbecs.reports.BloodAvailability.BloodAvailabilityReport();
			report.SetDataSource(dsReport);
			// Populating parameters of report.
			report.SetParameterValue("ReportTitle",reportTitle);
			report.SetParameterValue("UserName",printedBy);
			report.SetParameterValue("Division",new BOL.Division(Common.LogonUser.LogonUserDivisionCode).DivisionName);
			//
			DataSet dsHeader =  BOL.VbecsReport.GetReportHeader(printedBy);
			report.OpenSubreport("ReportHeader.rpt").SetDataSource(dsHeader);
			//
			string reportCriteria = this.ReportCriteria;
			DataSet dsFooter = BOL.VbecsReport.GetReportFooter(reportCriteria);
			report.OpenSubreport("ReportFooter.rpt").SetDataSource(dsFooter);
			*/
            return reportTitle;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/29/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="7372"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7373"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GenerateAndPrintReport
		/// </summary>
        protected override VbecsReportPrinter createReportPrinter()
		{
            DataSet dsReport = this.GetBloodAvailabilityReport();
            BOL.VbecsUser user = new BOL.VbecsUser(this.PrintedBy);
            DataSet dsHeader = BOL.VbecsReport.GetReportHeader(user.UserName);
            DataSet dsFooter = BOL.VbecsReport.GetReportFooter(this.ReportCriteria);

            ReportDataSource reportSource = new ReportDataSource();
            if (dsReport.Tables.Count > 0)
            {
                reportSource.Name = "BloodAvailabilityDataSet";
                reportSource.Value = dsReport.Tables[0];
            }

            ReportDataSource reportSourceHeader = new ReportDataSource();
            if (dsHeader.Tables.Count > 0)
            {
                reportSourceHeader.Name = "ReportHeaderDataSet";
                reportSourceHeader.Value = dsHeader.Tables[0];
            }

            ReportDataSource reportSourceFooter = new ReportDataSource();
            if (dsFooter.Tables.Count > 0)
            {
                reportSourceFooter.Name = "ReportFooterDataSet";
                reportSourceFooter.Value = dsFooter.Tables[0];
            }

            Assembly assembly = Assembly.LoadFrom(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location) + "\\Reports.dll");
            Stream stream = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.BloodAvailability.BloodAvailabilityReport.rdlc");

            LocalReport report = new LocalReport();
            report.DataSources.Add(reportSource);
            report.DataSources.Add(reportSourceHeader);
            report.DataSources.Add(reportSourceFooter);
            report.LoadReportDefinition(stream);
            report.SetParameters(new ReportParameter("ReportTitle", this.GenerateReportTitle()));

            return new BOL.VbecsReportPrinter(this.PrinterName, true, report);
		}

		#endregion
	}
}
